# 機能設計書 141-Web IDE

## 概要

本ドキュメントは、GitLabのWeb IDE機能に関する設計書である。Web IDEはブラウザ内で動作する統合開発環境であり、ユーザーがGitLabリポジトリのファイルを直接編集・作成・削除できる機能を提供する。

### 本機能の処理概要

Web IDEは、GitLabのリポジトリに対してブラウザ上からコードの編集作業を行うための統合開発環境である。従来のファイル単体編集とは異なり、複数ファイルの同時編集、リアルタイムプレビュー、ターミナルアクセスなど、ローカル開発環境に近い体験をWebブラウザ上で提供する。

**業務上の目的・背景**：開発者がローカル環境をセットアップすることなく、ブラウザから直接コードを編集・コミットできる環境を提供することで、小規模な変更や緊急修正時の作業効率を向上させる。また、環境構築が困難な状況や、別のマシンからの作業時にも開発作業を継続可能にする。

**機能の利用シーン**：
- READMEファイルやドキュメントの軽微な修正
- 設定ファイルの変更
- 緊急バグ修正時のホットフィックス対応
- コードレビュー中に発見した問題の即座の修正
- CI/CD設定ファイル（.gitlab-ci.yml）の編集・検証

**主要な処理内容**：
1. OAuth認証によるセキュアなアクセス制御
2. リポジトリファイルの読み込みと表示
3. ファイル編集・作成・削除の実行
4. 変更内容のコミットとプッシュ
5. フォークされたリポジトリでの作業サポート

**関連システム・外部連携**：
- OAuth2認証システム（WebIde::DefaultOauthApplication）
- Gitリポジトリバックエンド（Gitaly）
- フロントエンドのVue.jsアプリケーション

**権限による制御**：
- プロジェクトへの読み取り権限（:read_project）が必要
- プッシュ権限がない場合はフォーク経由での編集を案内
- フォーク権限がある場合はフォーク作成への誘導

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 277 | Web IDE | 主機能 | Web IDEエディタ |
| 277 | Web IDE | 補助機能 | ファイル編集 |
| 277 | Web IDE | 補助機能 | ファイル作成 |
| 278 | OAuth リダイレクト | 主機能 | Web IDE OAuthリダイレクト |

## 機能種別

ファイル操作 / コード編集 / バージョン管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| project_id | String | Yes | プロジェクトのフルパス（namespace/project形式） | 存在するプロジェクトであること |
| branch | String | No | 編集対象のブランチ名 | 有効なブランチ名であること |

### 入力データソース

- URLパラメータからプロジェクトID・ブランチ名を取得
- OAuth認証情報（セッション）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| レイアウト | HTML | fullscreenレイアウトで描画されたWeb IDE画面 |
| fork_info | Hash | フォーク関連情報（ide_path または fork_path） |

### 出力先

- ブラウザ画面（フルスクリーンレイアウト）
- セッションストレージ（OAuth認証情報）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信（GET /-/ide/project/:project_id）
   └─ ルーティングからIdeController#indexへディスパッチ

2. 認証・認可チェック
   └─ before_action :authorize_read_project!
   └─ プロジェクトの読み取り権限を確認

3. OAuthアプリケーション確認
   └─ before_action :ensure_web_ide_oauth_application!
   └─ WebIde::DefaultOauthApplication.ensure_oauth_application!

4. フォーク情報の取得
   └─ fork_info(project, params[:branch])
   └─ プッシュ権限の有無を確認
   └─ フォークの存在確認または作成パス生成

5. レンダリング
   └─ render layout: 'fullscreen'
   └─ フロントエンドのVue.jsアプリケーションを読み込み
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{プロジェクト存在確認}
    B -->|存在しない| C[404エラー]
    B -->|存在する| D{読み取り権限確認}
    D -->|権限なし| E[アクセス拒否]
    D -->|権限あり| F{削除保留中?}
    F -->|Yes| G[アクセス拒否]
    F -->|No| H[OAuthアプリ確認]
    H --> I{プッシュ権限確認}
    I -->|あり| J[直接編集モード]
    I -->|なし| K{既存フォーク確認}
    K -->|あり| L[フォーク編集パス生成]
    K -->|なし| M{フォーク権限確認}
    M -->|あり| N[フォーク作成パス生成]
    M -->|なし| O[閲覧のみモード]
    J --> P[Web IDE描画]
    L --> P
    N --> P
    O --> P
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-141-01 | プロジェクト読み取り権限 | Web IDEを開くにはプロジェクトの読み取り権限が必要 | 常時 |
| BR-141-02 | 削除保留プロジェクト除外 | 削除保留中のプロジェクトではWeb IDEを開けない | プロジェクトが削除保留状態の場合 |
| BR-141-03 | フォーク誘導 | プッシュ権限がない場合はフォーク経由での編集を案内 | プッシュ権限がない場合 |
| BR-141-04 | OAuthアプリケーション必須 | Web IDEの動作にはOAuthアプリケーションの設定が必要 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| プロジェクト検索 | projects | SELECT | フルパスからプロジェクトを検索 |
| フォーク検索 | projects | SELECT | ユーザーの既存フォークを検索 |
| OAuthアプリ確認 | oauth_applications | SELECT/INSERT | Web IDE用OAuthアプリの確認・作成 |

### テーブル別操作詳細

#### projects

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | id, name, path, namespace_id, pending_delete | path_with_namespace = params[:project_id] | プロジェクト特定 |

#### oauth_applications

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT/INSERT | id, name, redirect_uri | name = 'GitLab Web IDE' | Web IDE用OAuthアプリ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | プロジェクトが存在しない | エラーページ表示 |
| 403 | Forbidden | 読み取り権限がない | アクセス拒否ページ表示 |
| 404 | Not Found | OAuthアプリケーションが未設定 | oauth_redirect時に404を返す |

### リトライ仕様

- OAuthアプリケーションの作成に失敗した場合、次回アクセス時に再作成を試みる

## トランザクション仕様

- OAuthアプリケーションの作成は単一トランザクション内で実行
- Web IDE自体の操作はGitalyを経由したGit操作として処理

## パフォーマンス要件

- 画面初期表示: 3秒以内
- ファイル読み込み: 1秒以内（通常サイズのファイル）
- urgency: low（低優先度）

## セキュリティ考慮事項

- OAuth2認証によるセキュアなアクセス制御
- CSP（Content Security Policy）の適用（WebIdeCSP concern）
- 静的オブジェクトの外部ストレージCSP対応（StaticObjectExternalStorageCSP concern）
- プロジェクトの削除保留状態チェックによる不正アクセス防止
- 内部イベントトラッキング（track_internal_event 'web_ide_viewed'）

## 備考

- フロントエンドはVue.jsで実装されており、本コントローラはエントリーポイントを提供
- Feature Flag: build_service_proxy, reject_unsigned_commits_by_gitlab がフロントエンドに渡される
- 将来的にはOAuthアプリケーション未設定時のエラーハンドリング改善が予定されている（Issue #433322）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エントリーポイントを理解する

メインコントローラの処理フローを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ide_controller.rb | `app/controllers/ide_controller.rb` | Web IDEのエントリーポイント、認可処理、フォーク情報生成 |

**主要処理フロー**:
1. **10行目**: before_action :authorize_read_project! - プロジェクト読み取り権限の確認
2. **11行目**: before_action :ensure_web_ide_oauth_application! - OAuthアプリ確認
3. **24-28行目**: index アクション - フォーク情報取得とレンダリング
4. **30-37行目**: oauth_redirect アクション - OAuth認証後のリダイレクト処理
5. **57-69行目**: fork_info メソッド - プッシュ権限に応じたフォーク情報生成

**読解のコツ**:
- `RoutableActions`モジュールの`find_routable!`メソッドがプロジェクト検索の中核
- `auth_proc`で削除保留中のプロジェクトを除外している点に注目

#### Step 2: OAuthアプリケーション管理を理解する

Web IDE専用のOAuthアプリケーション管理機構を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | default_oauth_application.rb | `app/models/web_ide/default_oauth_application.rb` | OAuthアプリの自動作成・管理 |

**読解のコツ**:
- `ensure_oauth_application!`メソッドが存在しない場合に自動作成する仕組み
- シングルトンパターンでWeb IDE用OAuthアプリを管理

#### Step 3: CSPとセキュリティ設定を理解する

Web IDEのセキュリティ設定を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | web_ide_csp.rb | `app/controllers/concerns/web_ide_csp.rb` | Content Security Policy設定 |

### プログラム呼び出し階層図

```
IdeController
    │
    ├─ authorize_read_project!
    │      └─ find_routable!(Project, ...)
    │             └─ auth_proc (削除保留チェック)
    │
    ├─ ensure_web_ide_oauth_application!
    │      └─ WebIde::DefaultOauthApplication.ensure_oauth_application!
    │
    ├─ fork_info(project, branch)
    │      ├─ can?(:push_code, project)
    │      ├─ current_user.fork_of(project)
    │      └─ can?(:fork_project, project)
    │
    └─ render layout: 'fullscreen'
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL パラメータ     ───▶  IdeController#index      ───▶  フルスクリーンHTML
(project_id,              ├─ プロジェクト検索            (Web IDE UI)
 branch)                  ├─ 権限チェック
                          ├─ OAuthアプリ確認
                          └─ フォーク情報生成
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ide_controller.rb | `app/controllers/ide_controller.rb` | コントローラ | Web IDEのエントリーポイント |
| default_oauth_application.rb | `app/models/web_ide/default_oauth_application.rb` | モデル | OAuthアプリケーション管理 |
| web_ide_csp.rb | `app/controllers/concerns/web_ide_csp.rb` | Concern | CSP設定 |
| routable_actions.rb | `app/controllers/concerns/routable_actions.rb` | Concern | ルーティング可能オブジェクト検索 |
| static_object_external_storage_csp.rb | `app/controllers/concerns/static_object_external_storage_csp.rb` | Concern | 外部ストレージCSP |
| product_analytics_tracking.rb | `app/controllers/concerns/product_analytics_tracking.rb` | Concern | アナリティクストラッキング |
| fullscreen.html.haml | `app/views/layouts/fullscreen.html.haml` | ビュー | フルスクリーンレイアウト |
